const express = require('express');
const cors = require('cors');
const bodyParser = require('body-parser');
const mysql = require('mysql2/promise');
const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');

const app = express();
const PORT = process.env.PORT || 3000;
const JWT_SECRET = process.env.JWT_SECRET || 'your_jwt_secret_key';

// Middleware
app.use(cors());
app.use(bodyParser.json());
app.use(bodyParser.urlencoded({ extended: true }));


// Middleware untuk melayani file statis
app.use(express.static('public'));




// Database connection pool
const pool = mysql.createPool({
  host: '38.46.219.162',
  user: 'drimamskom',
  password: 'D15n4kj4t1m', // Replace with actual password in production
  port: 30012,
  database: 'school_management',
  waitForConnections: true,
  connectionLimit: 10,
  queueLimit: 0
});

// Authentication middleware
const authenticateToken = (req, res, next) => {
  const authHeader = req.headers['authorization'];
  const token = authHeader && authHeader.split(' ')[1];
  
  if (!token) return res.status(401).json({ message: 'Access denied' });
  
  jwt.verify(token, JWT_SECRET, (err, user) => {
    if (err) return res.status(403).json({ message: 'Invalid or expired token' });
    req.user = user;
    next();
  });
};

// Admin authentication middleware
const isAdmin = (req, res, next) => {
  if (req.user.role !== 'admin') {
    return res.status(403).json({ message: 'Admin access required' });
  }
  next();
};

// Routes
// Auth routes
app.post('/api/auth/login', async (req, res) => {
  try {
    const { email, password } = req.body;
    
    const [users] = await pool.query(
      'SELECT * FROM users WHERE email = ?',
      [email]
    );
    
    if (users.length === 0) {
      return res.status(401).json({ message: 'Invalid email or password' });
    }
    
    const user = users[0];
    const validPassword = await bcrypt.compare(password, user.password);
    
    if (!validPassword) {
      return res.status(401).json({ message: 'Invalid email or password' });
    }
    
    // Create token
    const token = jwt.sign(
      { id: user.id, email: user.email, role: user.role },
      JWT_SECRET,
      { expiresIn: '24h' }
    );
    
    // Remove password from response
    delete user.password;
    
    res.json({ token, user });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// User routes
app.get('/api/users', authenticateToken, isAdmin, async (req, res) => {
  try {
    const [users] = await pool.query(
      'SELECT id, name, email, role, profile_image, created_at, updated_at FROM users'
    );
    res.json(users);
  } catch (error) {
    console.error('Error fetching users:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

app.post('/api/users', authenticateToken, isAdmin, async (req, res) => {
  try {
    const { name, email, password, role } = req.body;
    
    // Hash password
    const salt = await bcrypt.genSalt(10);
    const hashedPassword = await bcrypt.hash(password, salt);
    
    const [result] = await pool.query(
      'INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)',
      [name, email, hashedPassword, role]
    );
    
    res.status(201).json({ id: result.insertId, name, email, role });
  } catch (error) {
    console.error('Error creating user:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Student routes
app.get('/api/students', authenticateToken, async (req, res) => {
  try {
    const [students] = await pool.query(
      'SELECT * FROM students'
    );
    res.json(students);
  } catch (error) {
    console.error('Error fetching students:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

app.get('/api/students/:id', authenticateToken, async (req, res) => {
  try {
    const [students] = await pool.query(
      'SELECT * FROM students WHERE id = ?',
      [req.params.id]
    );
    
    if (students.length === 0) {
      return res.status(404).json({ message: 'Student not found' });
    }
    
    res.json(students[0]);
  } catch (error) {
    console.error('Error fetching student:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Grades routes
app.get('/api/grades', authenticateToken, async (req, res) => {
  try {
    const { studentId } = req.query;
    let query = 'SELECT * FROM grades';
    let params = [];
    
    if (studentId) {
      query += ' WHERE student_id = ?';
      params.push(studentId);
    }
    
    const [grades] = await pool.query(query, params);
    res.json(grades);
  } catch (error) {
    console.error('Error fetching grades:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Attendance routes
app.get('/api/attendance', authenticateToken, async (req, res) => {
  try {
    const { studentId } = req.query;
    let query = 'SELECT * FROM attendance';
    let params = [];
    
    if (studentId) {
      query += ' WHERE student_id = ?';
      params.push(studentId);
    }
    
    const [attendance] = await pool.query(query, params);
    res.json(attendance);
  } catch (error) {
    console.error('Error fetching attendance:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Assignments routes
app.get('/api/assignments', authenticateToken, async (req, res) => {
  try {
    const { courseId } = req.query;
    let query = 'SELECT * FROM assignments';
    let params = [];
    
    if (courseId) {
      query += ' WHERE course_id = ?';
      params.push(courseId);
    }
    
    const [assignments] = await pool.query(query, params);
    res.json(assignments);
  } catch (error) {
    console.error('Error fetching assignments:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Messages routes
app.get('/api/messages', authenticateToken, async (req, res) => {
  try {
    const { userId } = req.query;
    
    const [messages] = await pool.query(
      'SELECT * FROM messages WHERE sender_id = ? OR receiver_id = ? ORDER BY timestamp DESC',
      [userId, userId]
    );
    
    res.json(messages);
  } catch (error) {
    console.error('Error fetching messages:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

app.post('/api/messages', authenticateToken, async (req, res) => {
  try {
    const { senderId, receiverId, content } = req.body;
    
    const [result] = await pool.query(
      'INSERT INTO messages (sender_id, receiver_id, content) VALUES (?, ?, ?)',
      [senderId, receiverId, content]
    );
    
    const [newMessage] = await pool.query(
      'SELECT * FROM messages WHERE id = ?',
      [result.insertId]
    );
    
    res.status(201).json(newMessage[0]);
  } catch (error) {
    console.error('Error sending message:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Schedule routes
app.get('/api/schedule', authenticateToken, async (req, res) => {
  try {
    const [schedule] = await pool.query('SELECT * FROM schedule');
    res.json(schedule);
  } catch (error) {
    console.error('Error fetching schedule:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Fees routes
app.get('/api/fees', authenticateToken, async (req, res) => {
  try {
    const { studentId } = req.query;
    let query = 'SELECT * FROM fees';
    let params = [];
    
    if (studentId) {
      query += ' WHERE student_id = ?';
      params.push(studentId);
    }
    
    const [fees] = await pool.query(query, params);
    res.json(fees);
  } catch (error) {
    console.error('Error fetching fees:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Documents routes
app.get('/api/documents', authenticateToken, async (req, res) => {
  try {
    const { userId } = req.query;
    let query = 'SELECT * FROM documents';
    let params = [];
    
    if (userId) {
      query += ' WHERE user_id = ?';
      params.push(userId);
    }
    
    const [documents] = await pool.query(query, params);
    res.json(documents);
  } catch (error) {
    console.error('Error fetching documents:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Admin dashboard routes
app.get('/api/admin/dashboard', authenticateToken, isAdmin, async (req, res) => {
  try {
    // Get counts for dashboard
    const [userCount] = await pool.query('SELECT COUNT(*) as count FROM users');
    const [studentCount] = await pool.query('SELECT COUNT(*) as count FROM students');
    const [courseCount] = await pool.query('SELECT COUNT(*) as count FROM courses');
    const [messageCount] = await pool.query('SELECT COUNT(*) as count FROM messages');
    
    res.json({
      userCount: userCount[0].count,
      studentCount: studentCount[0].count,
      courseCount: courseCount[0].count,
      messageCount: messageCount[0].count
    });
  } catch (error) {
    console.error('Error fetching dashboard data:', error);
    res.status(500).json({ message: 'Server error' });
  }
});

// Start server
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

module.exports = app;